#!/usr/bin/python
#####################################################################################
# k4cal.py:  Perform Low Power (5W) and High Power (50W) TX Gain Calibration on a K4
#            via telnet commands
#
# Author:   N6TV
#
# Time-stamp: "19 March 2021 07:52 UTC"
#
# Method:
#    Cycle through all bands once at low power (5W), once at high power (50W) and tap TUNE
#
# Usage:
#     k4cal.py [IP address of K4] [Dummy Antenna Port 1 to 3] [debug]
#     python k4cal.py [IP address of K4] [Dummy Antenna Port 1 to 3] [debug]
#
# Input:
#    IP Address of K4
#
# Output:
#    K4 xxxx Gain Cal YYYYMMDD HHMMz.txt - Report with S/N xxxx and timestamp in file name
#    STDOUT - display of old and new TX Gain Cal. settings for each band and power level
#
# Notes:
#    Attempts to save and restore prior power, antenna, split, and frequency settings on
#    each band.  Covers 160m to 6m only.
#
# Change Log:
#    2020-11-07 - 0.94 - Initial release of compiled Windows .exe version
#    2020-11-12 - 0.95 - Save report to file with S/N and Date Timestamp (tnx N6XI)
#    2020-11-20 - 0.96 - Use SN; command to get 5-digit radio serial no. instead of ME33;
#                        (tnx W6FVI)
#    2021-02-05 - 0.97 - Hold TUNE for 4 seconds instead of 1.25 seconds to allow more time
#                        for power to settle
#                      - Increase read timeouts
#                      - Use new PCX; command (Ksrv 1146 or later) to query current
#                        power level
#                      - Calibration frequency changes
#                        1850 kHz instead of 1900 kHz
#                        28500 kHz insteads of 29000 kHz
#    2021-03-19 - 0.98 - Instead of 4 second fixed tune, Sample power every 400 ms after
#                        key down, up to 20 samples, stopping
#                        when there are four consecutive identical samples in range.
#                      - Bypass ATU during tune, restore upon completion.
#                      - Test SWR after every band change.  Terminate calibration if
#                        SWR higher than 1.5:1.
#
# Language:  Python 2.7 or later
#####################################################################################
import sys                              # For exit(), argv
import os.path                          # For basename
import telnetlib                        # For telnet access
import time                             # For sleep
import __builtin__                      # To override raw_input to use stderr
from collections import namedtuple
from datetime import datetime

#
# Constants
#
debug = False

VERSION = "0.98"

PORT = 9200                             # K4 default TELNET port

#CAL_FREQS = {                           # Calibration frequencies for CW ops
#    0 : 1825,
#    1 : 3540,
#    2 : 5336,
#    3 : 7040,
#    4 : 10125,
#    5 : 14040,
#    6 : 18110,
#    7 : 21040,
#    8 : 24930,
#    9 : 28045,
#   10 : 51100
#}

CAL_FREQS = {                           # Calibration frequencies used by K3 Utility
    0 : 1850,                           # K3 Utility used 1900
    1 : 3750,
    2 : 5336,
    3 : 7150,
    4 : 10125,
    5 : 14200,
    6 : 18110,
    7 : 21200,
    8 : 24930,
    9 : 28500,                          # K3 Utility used 29000
   10 : 52000
}

#
# Arrays and dictionaries
#

# Acceptable tune power readings:
#   4.9, 5.0, or 5.1 W for low power cal
#   49, 50, or 51 W for high power cal                           
PWROK = { 'L' : ['PO0049;', 'PO0050;', 'PO0051;'],
          'H' : ['PO0490;', 'PO0500;', 'PO0510;']
}

bandInfo = namedtuple("bandInfo", "tuneFreq oldGainLP newGainLP oldGainHP newGainHP")
bands = {}                             # Key = Band No., 0 to 10
                                       # Value = bandInfo(tuneFreq,  oldGainLP, newGainLP,
                                       #                  olddGainHP, newGainHP)

#
# Functions
#
def raw_input(prompt=None):
    if prompt:
        sys.stderr.write(str(prompt))
    return __builtin__.raw_input()

# Send command to radio, with optional tracing
def SendCmd(tn, cmd):
    if debug:
        print >> sys.stderr, "<<", cmd
    if cmd is not None:
        tn.write(cmd)

def ReadResult(tn, timeout):
    result = tn.read_until(";", timeout)
    if debug:
        print >> sys.stderr, ">>", result
    if result in (None, ''):
        print >> sys.stderr, "Timed out waiting for reply."
    return result

# System-dependent "Pause" command
def OsPause():
    if os.name == 'nt':                 # Windows
        # Redirect prompt to stderr
        os.system('Pause 1>&2')
    elif os.name == 'posix':            # Unix
        os.system('read -n1 -r -p "Press any key to continue . . ."')

# System-dependent "Break" command
def OsBreak():
    if os.name == 'nt':                 # Windows
        return "Ctrl-Break"
    else:
        return "Ctrl-C"

# Display variable value on STDERR
def Trace(varname, value):
    if debug:
        print >> sys.stderr, varname, "=", value
        
#
# Save original radio configuration (frequency, antenna, power, split, gain cal via tune option,
# and test mode.
#
def SaveCfg():
    global origFreq, origAnt, origSplit, origGainCalViaTune, origTestMode
    #
    # Save original VFO frequency which will be restored at end of calibration
    #
    SendCmd(tn, "FA;")
    origFreq = ReadResult(tn, 2);
    
    Trace("origFreq", origFreq)
    
    #
    # Save original ANT setting which will be restored at end of calibration
    #
    SendCmd(tn, "AN;")
    origAnt = ReadResult(tn, 2);
    
    Trace("origAnt", origAnt)
    
    #
    # Save original SPLIT state
    #
    SendCmd(tn, "FT;")
    origSplit = ReadResult(tn, 2)
    
    Trace("origSplit", origSplit)
    
    #
    # Save original state of TX Gain Cal via TUNE
    #
    SendCmd(tn, "ME15;")
    origGainCalViaTune = ReadResult(tn, 2);
    
    Trace("origGainCalViaTune", origGainCalViaTune)
    
    #
    # Save original state of TEST mode
    #
    SendCmd(tn, "TS;")
    origTestMode = ReadResult(tn, 2);
    
    Trace("origGainCalViaTune", origGainCalViaTune)

#
# Restore original radio configuration (frequency, antenna, power, split, gain cal via tune
# option, and test mode.
#
def RestoreCfg():
    SendCmd(tn, origFreq + origAnt + origSplit + origGainCalViaTune + origTestMode)
    
    # Can't close TELNET connection or exit until commands complete
    time.sleep(3)

#
# Save current frequency, power, and antenna being used on current band
# 
def SaveBandCfg():
    global oldFreq, oldPwr, oldAnt
    
    # Read current frequency on this band
    SendCmd(tn, "FA;")
    oldFreq = ReadResult(tn, 2)

    # Get current power level on this band
    SendCmd(tn, "PCX;")
    pwr = ReadResult(tn, 3)
    # Returns PC052L; for 5.2 Watts, PC052H: for 52 Watts.
    #         ----+--
    if len(pwr) == 7 and pwr[0:2] == "PC" and pwr[5] in ['L', 'l', 'H', 'h'] and pwr[6] == ';':
        oldPwr = pwr;
    else:
        print >> sys.stderr, "Could not read current power level.", "pwr =", pwr
        oldPwr = None

    # Read current antenna setting for this band
    SendCmd(tn, "AN;")
    oldAnt = ReadResult(tn, 2)

#
# Restore frequency, power, and antenna saved after changing to this band
#
def RestoreBandCfg():
    # Change frequency back to original;
    SendCmd(tn, oldFreq)

    # Change power back to original;
    SendCmd(tn, oldPwr);

    # Change antenna back to original
    SendCmd(tn, oldAnt);


def SWRisOK():
    SendCmd(tn, "DE040;TM;")
    # Returns:  TM005000051014;
    #           ----+----1----+
    result = False
    tm = ReadResult(tn, 2)
    if tm is not None and len(tm) == 15:
        # SWR 1.4 is last two digits '14'
        try:
            swr = int(tm[-3:-1])
        except (ValueError, TypeError):
            print >> sys.stderr, "Could not calculate current SWR.  TM; returned: '", tm, "'"
            
        # SWR must be between 1.0 and 1.5:1, inclusive)
        result = (swr >= 10 and swr <= 15)
    else:
        print >> sys.stderr, "Could not obtain current SWR.  TM; returned: '", tm, "'"

    return result
        
    
# Issue TUNE command, then read output power every 400 ms until
# four identical power readings within acceptable range are determined,
# but quit after 20 samples if not successful.
def TryTune(LorH):
    po = [None] * 4                     # List of 4 most recent power samples
    converged = False

    # Send tune comand
    SendCmd(tn, "TU1;")

    if SWRisOK():
        SendCmd(tn, "DE040;PO;")
        for i in range(0,20):
            pwr = ReadResult(tn, 2)
            # Remove last power reading from list
            del po[-1]
            
            # Insert new power reading to front of list
            po.insert(0, pwr)
   
            if debug:
                print >> sys.stderr, "po =", po
   
            # Are all four low power readings acceptable and identical?
            if po[0] in PWROK.get(LorH, []) and po[:-1] == po[1:]:
                converged = True
                if debug:
                    print >> sys.stderr, "Power converged with i =", i
                # Leave for loop
                break
   
            # Haven't converged yet, get another power reading afer 300 ms
            SendCmd(tn, "DE030;PO;")
    else:
        print >> sys.stderr, "SWR too high."
   
    # Stop tuning
    SendCmd(tn, "DE030;TU0;")

    return converged

####################
# Mainline start
####################
debug = False                           # Prints extra debugging traces
host = ""
dummyAnt = ""

#
# Read command line args, if any
#
argc = len(sys.argv)
if argc >= 2:
    if sys.argv[1].lower() in ["?", "-?", "-h", "--help"]:
        print >> sys.stderr, \
              "Usage: ", os.path.basename(sys.argv[0]), "[IP-address] [Dummy-Ant-Port] [debug]"
        sys.exit(4)

    host = sys.argv[1]
    if argc >= 3:
        dummyAnt = sys.argv[2]
    if argc >= 4:
        debug = sys.argv[3].upper() == "DEBUG"

print >> sys.stderr, "K4 TX Gain Calibration Version", VERSION, "by N6TV"
print >> sys.stderr, "Press", OsBreak(), "to interrupt at any point."

#
# Prompt for IP address of K4 if not specified as arg, and connect
#

if host == "":
    host = raw_input("Enter IP address of K4:  ")

if host.strip() == "":
    print >> sys.stderr, "IP address not specified - terminating."
    sys.exit(4)
    
print >> sys.stderr, "Connecting to", host, "port", PORT, "via TELNET ..."

try:
    tn = telnetlib.Telnet(host, PORT, 5)    # 5 second timeout if can't connect
except:
    print >> sys.stderr, "Could not connect to", host, "port", PORT, "- terminating."
    sys.exit(4)

#
# Get K4 serial no. for report
#
SendCmd(tn, "SN;")
# Returns SNnnnnn; where nnnn = K4 Serial No.
result = ReadResult(tn, 2)
serial = result[2:7]

print >> sys.stderr, "Connect to K4 S/N", serial, "successful."

# Make sure PCX; command supported, requires ksrv 1146 or later
SendCmd(tn, "PCX;")
pwr = ReadResult(tn, 2)
# Returns PC052L; for 5.2 Watts, PC052H: for 52 Watts.
#         ----+--
if len(pwr) == 7 and pwr[0:2] == "PC" and pwr[5] in ['L', 'l', 'H', 'h'] and pwr[6] == ';':
    pass
else:
    print >> sys.stderr, "Could not read current power level with PCX; command"
    print >> sys.stderr, "This program requires ksrv version 1146 or later. Terminating."
    # Early exit
    sys.exit(8)

if dummyAnt == "":
    dummyAnt = raw_input("Enter antenna port of 50 ohm 100W dummy load (1-3):  ")

if dummyAnt not in ["1", "2", "3"]:
    print >> sys.stderr, "Antenna no. '" + dummyAnt + "' not recognized, terminating."
    sys.exit(4)

# Save original radio configuration (frequency, antenna, power, split, gain cal via tune option,
# and test mode.
SaveCfg()

print >> sys.stderr, "Ready to start TX Gain Calibration Procedure using dummy load on Ant", dummyAnt, "?"
OsPause()

# Enable TX Gain Cal via TUNE;
SendCmd(tn, "DE020;ME15.0001;")

# Turn SPLIT OFF
SendCmd(tn, "FT0;")

# Turn off TEST mode
SendCmd(tn, "TS0;")

#
# Intialize bands array
#
for band in range (0, 11):
    bands[band] = None

#
# Loop through Bands 0 to 10 (160m through 6m)
#
for band in range (0, 11):
    SendCmd(tn, "BN%02d;" % band)
    time.sleep(1)

    # Save current frequency, power, and antenna being used on current band
    SaveBandCfg()

    # Switch to dummy load antenna
    SendCmd(tn, "AN%s;" % dummyAnt)

    # Read current antenna tuner state for this band and antenna
    SendCmd(tn, "DE050;AT;")
    oldATU = ReadResult(tn, 2)

    # Bypass the ATU
    SendCmd(tn, "AT1;")

    # Set power 5W
    SendCmd(tn, "PC050L;")
    time.sleep(1)

    #
    # Read current Low Power TX Gain Cal
    #
    SendCmd(tn, "TG;")

    # Returns TGnnnn;
    try:
        oldGainLP = int(ReadResult(tn, 3)[2:6])
    except (ValueError, TypeError):
        print >> sys.stderr, "Could not read current 5W TX Gain.", "oldGainLP =", oldGainLP
        oldGainLP = 0

    # Change frequency to tune Freq
    tuneFreq = CAL_FREQS[band]
    SendCmd(tn, "FA%d;" % tuneFreq)
    time.sleep(1)

    print >> sys.stderr, "Calibrating  5W at", tuneFreq, "kHz"

    # Issue TUNE command and wait for power output to stabilize at accepted values
    converged = TryTune('L')

    if converged == False:
        print >> sys.stderr, "Power output did not converge, terminating."
        RestoreBandCfg()
        RestoreCfg()
        sys.exit(8);

    # Read new Low Power TX Gain Cal
    SendCmd(tn, "DE100;TG;")
    try:
        newGainLP = int(ReadResult(tn, 7)[2:6])
    except (ValueError, TypeError):
        print >> sys.stderr, "Could not read new 5W TX Gain"
        newGainLP = 0

    # Set power to 50W
    SendCmd(tn, "PC050H;")
    time.sleep(1)

    #
    # Read current High Power TX Gain Cal
    #
    SendCmd(tn, "TG;")
    try:
        oldGainHP = int(ReadResult(tn, 2)[2:6])
    except (ValueError, TypeError):
        print >> sys.stderr, "Could not read old 50W TX Gain"
        oldGainHP = 0

    print >> sys.stderr, "Calibrating 50W at", tuneFreq, "kHz"

    # Issue TUNE command and wait for power output to stabilize at accepted values
    converged = TryTune('H')

    if converged == False:
        print >> sys.stderr, "Power output did not converge, terminating."
        RestoreBandCfg()
        RestoreCfg()
        sys.exit(8);

    # Read new High Power TX Gain Cal
    SendCmd(tn, "DE100;TG;")
    try:
        newGainHP = int(ReadResult(tn, 7)[2:6])
    except (ValueError, TypeError):
        print >> sys.stderr, "Could not read new 50W TX Gain.", "newGainHP = ", newGainHP
        newGainHP = 0

    bands[band] = bandInfo(tuneFreq, oldGainLP, newGainLP, oldGainHP, newGainHP)

    Trace("bands[" + str(band) + "]", bands[band])

    # Change ATU for this dummy load antenna back to old value
    SendCmd(tn, oldATU);

    # Restore frequency, power, and antenna saved after changing to this band
    RestoreBandCfg()

    time.sleep(1)

    # End for loop through bands

# Restore original radio configuration (frequency, antenna, power, split, gain cal via tune
# option, and test mode.
RestoreCfg()

#
# Close TELNET connection
#
tn.close()

#
# Write gain report to output file "K4 xxxx Gain Cal YYYYMMDD HHMMz.txt"
#

outFileName = "K4 %s Gain Cal %s.txt" % (serial, datetime.utcnow().strftime('%Y%m%d %H%Mz'))

outFile = open(outFileName, 'w')

outFile.write("TX Gain Calibration report for K4 S/N %s - %s\n\n" %
              (serial, datetime.utcnow().strftime('%Y-%m-%d %H:%M UTC')))
outFile.write("%6s%14s%14s%14s%14s%14s%14s\n" %
              ("Freq", "Old 5W Gain", "New 5W Gain", "5W Delta", "Old 50W Gain", "New 50W Gain", "50W Delta"))

for band in range (0, 11):
    info = bands[band]
    if info is not None:
        outFile.write("%6d%14d%14d% 14d%14d%14d% 14d\n" %
                      (info.tuneFreq,
                       info.oldGainLP,
                       info.newGainLP,
                       info.newGainLP - info.oldGainLP,
                       info.oldGainHP,
                       info.newGainHP,
                       info.newGainHP - info.oldGainHP))
        
outFile.write("\n%s version %s by N6TV\n" % (os.path.basename(sys.argv[0]), VERSION))
outFile.close()

# Display results on STDOUT
os.system('more "%s"' % outFileName)

print "This report saved to", outFileName
OsPause()
